<?php
/**
 * Creative Elements - live Theme & Page Builder
 *
 * @author    WebshopWorks
 * @copyright 2019-2023 WebshopWorks.com
 * @license   One domain support license
 */

namespace CE;

defined('_PS_VERSION_') or exit;

use CE\CoreXDynamicTagsXDataTag as DataTag;
use CE\ModulesXDynamicTagsXModule as Module;

class ModulesXCatalogXTagsXProductImages extends DataTag
{
    const REMOTE_RENDER = true;

    public function getName()
    {
        return 'product-images';
    }

    public function getTitle()
    {
        return __('Product Images');
    }

    public function getGroup()
    {
        return Module::CATALOG_GROUP;
    }

    public function getCategories()
    {
        return [Module::GALLERY_CATEGORY];
    }

    protected function _registerControls()
    {
        $this->addControl(
            'id_product',
            [
                'label' => __('Product'),
                'type' => ControlsManager::SELECT2,
                'label_block' => true,
                'select2options' => [
                    'placeholder' => __('Current Product'),
                    'ajax' => [
                        'get' => 'Products',
                        'url' => Helper::getAjaxProductsListLink(),
                    ],
                ],
            ]
        );

        $image_size_options = GroupControlImageSize::getAllImageSizes('products');

        $this->addControl(
            'image_size',
            [
                'label' => __('Image Size'),
                'label_block' => true,
                'type' => ControlsManager::SELECT,
                'options' => $image_size_options,
                'default' => key($image_size_options),
            ]
        );

        $this->addControl(
            'start',
            [
                'label' => __('Start'),
                'type' => ControlsManager::NUMBER,
                'min' => 1,
                'placeholder' => 1,
            ]
        );

        $this->addControl(
            'limit',
            [
                'label' => __('Limit'),
                'type' => ControlsManager::NUMBER,
                'min' => 1,
                'placeholder' => __('None'),
            ]
        );

        $this->addControl(
            'show_caption',
            [
                'label' => __('Caption'),
                'type' => ControlsManager::SWITCHER,
                'label_on' => __('Show'),
                'label_off' => __('Hide'),
                'default' => 'yes',
            ]
        );
    }

    public function getValue(array $options = [])
    {
        $context = \Context::getContext();
        $settings = $this->getSettings();
        $items = [];

        if (!$settings['id_product'] && $context->controller instanceof \ProductController) {
            $images = $context->smarty->tpl_vars['product']->value['images'];
        } else {
            $product = new \Product($settings['id_product'], false, $context->language->id);
            $imageRetriever = new \PrestaShop\PrestaShop\Adapter\Image\ImageRetriever($context->link);
            $settings['id_product_attribute'] = $product->cache_default_attribute;
            $images = $imageRetriever->getProductImages($settings, $context->language);
        }

        if (!empty($settings['start']) || !empty($settings['limit'])) {
            $images = array_slice($images, ($settings['start'] ?: 1) - 1, $settings['limit'] ?: null);
        }

        foreach ($images as &$image) {
            $bySize = $image['bySize'][$settings['image_size']];
            $items[] = [
                'image' => [
                    'id' => '',
                    'url' => $bySize['url'],
                    'alt' => $image['legend'],
                    'width' => $bySize['width'],
                    'height' => $bySize['height'],
                ],
                'link' => [
                    'url' => Helper::getProductImageLink($image),
                ],
                'caption' => $settings['show_caption'] ? $image['legend'] : '',
            ];
        }

        return $items;
    }
}
