<?php
/**
 * Copyright (c) since 2010 Stripe, Inc. (https://stripe.com)
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License version 3.0
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    Stripe <https://support.stripe.com/contact/email>
 * @copyright Since 2010 Stripe, Inc.
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License version 3.0
 */

use Stripe\Checkout\Session;
use Stripe\PaymentIntent;

if (!defined('_PS_VERSION_')) {
    exit;
}

class PrestashopOrderService
{
    private $context;
    /**
     * @var Stripe_official
     */
    private $module;

    private $psStatuses;
    /**
     * @var StripePaymentMethodService
     */
    private $stripePaymentMethodService;
    /**
     * @var StripePaymentIntentService
     */
    private $stripePaymentIntentService;
    /**
     * @var StripeCheckoutSessionService
     */
    private $stripeCheckoutSessionService;

    public function __construct($context, $module, $secretKey)
    {
        $this->context = $context;
        $this->module = $module;
        $secretKey = $secretKey ?: Stripe_official::getSecretKey();
        $this->stripePaymentMethodService = new StripePaymentMethodService($secretKey);
        $this->stripePaymentIntentService = new StripePaymentIntentService($secretKey);
        $this->stripeCheckoutSessionService = new StripeCheckoutSessionService($secretKey);
    }

    public function createPsOrder(OrderModel $orderModel): OrderModel
    {
        $orderId = Order::getIdByCartId($orderModel->cartId);
        $order = null;
        if ($orderId) {
            $order = new Order($orderId);
        }

        if (!$order && (int) $orderModel->status) {
            try {
                $this->module->validateOrder(
                    $orderModel->cartId,
                    (int) $orderModel->status,
                    $orderModel->amount,
                    sprintf(
                        self::useNewTranslationSystem() ? $this->module->getTranslator()->trans('%s via Stripe', [], 'Modules.Stripeofficial.PrestashopOrderService') : $this->module->l('%s via Stripe', 'PrestashopOrderService'),
                        Tools::ucfirst($orderModel->paymentMethodType)
                    ),
                    $orderModel->message,
                    [],
                    null,
                    false,
                    $orderModel->secureKey,
                    $orderModel->shop
                );

                $orderId = Order::getIdByCartId($orderModel->cartId);
                if ($orderId) {
                    $order = new Order($orderId);
                }
            } catch (Exception $e) {
            }
        }
        $orderId = $orderId ?: null;
        $order = $order ?: null;
        $reference = $order ? $this->context->shop->name . ' ' . $order->reference : null;
        $orderModel->orderId = $orderId;
        $orderModel->order = $order;
        $orderModel->orderReference = $reference;

        return $orderModel;
    }

    public function getOrderConfirmationLink(OrderModel $orderModel)
    {
        return $this->context->link->getPageLink(
            'order-confirmation',
            true,
            null,
            [
                'id_cart' => $orderModel->cartId ?: 0,
                'id_module' => (int) $this->module->id,
                'id_order' => $orderModel->orderId,
                'key' => $orderModel->secureKey,
            ]
        );
    }

    public function buildOrderModel(
        StripePaymentIntent $psStripePaymentIntent,
        PaymentIntent $stripePaymentIntent,
        ?Cart $cart = null
    ): OrderModel {
        $message = 'Stripe Transaction ID: ' . $stripePaymentIntent->id;
        $amount = Stripe_official::isZeroDecimalCurrency($stripePaymentIntent->currency) ?
            (float) $stripePaymentIntent->amount :
            (float) number_format($stripePaymentIntent->amount / 100, 2, '.', '')
        ;
        $stripePaymentMethod = $this->stripePaymentMethodService->getStripePaymentMethod($stripePaymentIntent->payment_method);
        $paymentMethodType = $this->stripePaymentMethodService->getStripePaymentMethodTypeByPaymentIntent($stripePaymentIntent);
        $orderStatus = $psStripePaymentIntent->getPsStatusForOrderCreation($paymentMethodType);

        $orderModel = new OrderModel();
        $orderModel->cart = $cart ?: ($this->context->cart ?: null);
        $orderModel->cartId = $cart ? $cart->id : ($this->context->cart ? $this->context->cart->id : null);
        $orderModel->secureKey = $cart ? $cart->secure_key : ($this->context->cart ? $this->context->cart->secure_key : null);
        $orderModel->shop = $this->context->shop;
        $orderModel->message = $message;
        $orderModel->amount = $amount;
        $orderModel->status = $orderStatus;
        $orderModel->paymentMethod = $stripePaymentMethod;
        $orderModel->paymentMethodType = $paymentMethodType;

        return $orderModel;
    }

    public function createPsStripePayment(PaymentIntent $stripePaymentIntent, OrderModel $orderModel): StripePayment
    {
        $stripePaymentMethodBillingDetails = $this->stripePaymentMethodService->getBillingDetailsFromStripePaymentMethod($orderModel->paymentMethod);
        $paymentOwnerName = isset($stripePaymentMethodBillingDetails->name) ? $stripePaymentMethodBillingDetails->name : '';
        $chargeId = (isset($stripePaymentIntent->charges->data->id) ?
            $stripePaymentIntent->charges->data->id :
            (
                isset($stripePaymentIntent->latest_charge) ?
                $stripePaymentIntent->latest_charge :
                null
            ));

        $stripePayment = new StripePayment();
        $stripePayment->setIdStripe($chargeId);
        $stripePayment->setIdPaymentIntent($stripePaymentIntent->id);
        $stripePayment->setName($paymentOwnerName);
        $stripePayment->setIdCart((int) $orderModel->cartId);

        $cardType = $orderModel->paymentMethodType;
        if (isset($orderModel->paymentMethod->card)) {
            $cardType = $orderModel->paymentMethod->card->brand;
        }

        $stripePayment->setType(Tools::strtolower($cardType));
        $stripePayment->setAmount($orderModel->amount);
        $stripePayment->setRefund(0);
        $stripePayment->setCurrency($stripePaymentIntent->currency);
        $stripePayment->setResult(1);
        $stripePayment->setState((int) Configuration::get('STRIPE_MODE'));
        $voucherUrl = isset($stripePaymentIntent->next_action->oxxo_display_details->hosted_voucher_url) ?
            $stripePaymentIntent->next_action->oxxo_display_details->hosted_voucher_url :
            null
        ;
        $voucherExpire = isset($stripePaymentIntent->next_action->oxxo_display_details->expires_after) ?
            date('Y-m-d H:i:s', $stripePaymentIntent->next_action->oxxo_display_details->expires_after) :
            null
        ;
        if ($voucherUrl && $voucherExpire) {
            $stripePayment->setVoucherUrl($voucherUrl);
            $stripePayment->setVoucherExpire($voucherExpire);
        }

        $stripePayment->setDateAdd(date('Y-m-d H:i:s'));
        $stripePayment->save();

        return $stripePayment;
    }

    public function findStripePaymentIntent(?string $paymentIntentId = null)
    {
        if (!$paymentIntentId) {
            return null;
        }

        $intent = $this->stripePaymentIntentService->getStripePaymentIntent($paymentIntentId);
        if (!$intent) {
            $session = $this->stripeCheckoutSessionService->getStripeCheckoutSession($paymentIntentId);
            if ($session instanceof Session) {
                $intent = $this->stripePaymentIntentService->getStripePaymentIntent($session->payment_intent);
            }
        }

        return $intent;
    }

    public static function useNewTranslationSystem()
    {
        if (version_compare(_PS_VERSION_, '1.7.6', '>=')) {
            return true;
        }

        return false;
    }
}
