<?php
/**
 * Copyright (c) since 2010 Stripe, Inc. (https://stripe.com)
 *
 * NOTICE OF LICENSE
 *
 * This source file is subject to the Academic Free License version 3.0
 * that is bundled with this package in the file LICENSE.md.
 * It is also available through the world-wide-web at this URL:
 * https://opensource.org/licenses/AFL-3.0
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@prestashop.com so we can send you a copy immediately.
 *
 * @author    Stripe <https://support.stripe.com/contact/email>
 * @copyright Since 2010 Stripe, Inc.
 * @license   https://opensource.org/licenses/AFL-3.0 Academic Free License version 3.0
 */

use StripeOfficial\Classes\StripeProcessLogger;

if (!defined('_PS_VERSION_')) {
    exit;
}

class stripe_officialOrderConfirmationReturnModuleFrontController extends ModuleFrontController
{
    /**
     * @var StripePaymentIntentService
     */
    private $stripePaymentIntentService;

    /**
     * @var PrestashopOrderService
     */
    private $prestashopOrderService;

    public function __construct($secretKey = null)
    {
        parent::__construct();
        $this->ssl = true;
        $this->ajax = true;
        $this->json = true;
        $secretKey = $secretKey ?: Stripe_official::getSecretKey();
        $this->stripePaymentIntentService = new StripePaymentIntentService($secretKey);
        $this->prestashopOrderService = new PrestashopOrderService($this->context, $this->module, $secretKey);
    }

    /**
     * @throws Exception
     *
     * @see FrontController::initContent()
     */
    public function initContent()
    {
        parent::initContent();

        $cartId = Tools::getValue('cartId');
        $failUrl = $redirectUrl = $this->context->link->getModuleLink('stripe_official', 'orderFailure', ['cartId' => $cartId], true);

        try {
            if (!$cartId) {
                Tools::redirect($failUrl);
            }

            $stripeIdempotencyKey = StripeIdempotencyKey::getOrCreateIdempotencyKey($cartId);
            if (!$stripeIdempotencyKey->id_payment_intent) {
                Tools::redirect($failUrl);
            }
            $intent = $this->prestashopOrderService->findStripePaymentIntent($stripeIdempotencyKey->id_payment_intent);
            if (!$intent) {
                Tools::redirect($failUrl);
            }

            $psStripePaymentIntent = new StripePaymentIntent();
            $psStripePaymentIntent->findByIdPaymentIntent($stripeIdempotencyKey->id_payment_intent);

            $status = null;
            $lastPaymentError = $intent->last_payment_error ?? null;
            if ($lastPaymentError) {
                $chargeDeclineCode = $lastPaymentError->decline_code ?? $lastPaymentError->code ?? null;
                $status = $psStripePaymentIntent->getStatusFromStripeDeclineCode($chargeDeclineCode);
                StripeProcessLogger::logInfo('Last Payment Error: ' . json_encode($lastPaymentError), 'orderConfirmationReturn', $cartId, $intent->id);
            }
            $status = $status ?? $psStripePaymentIntent->getStatusFromStripePaymentIntentStatus($intent->status);
            if ($psStripePaymentIntent->validateStatusChange($status)) {
                $psStripePaymentIntent->setIdPaymentIntent($intent->id);
                $psStripePaymentIntent->setAmount($intent->amount);
                $psStripePaymentIntent->setStatus($status);
                $psStripePaymentIntent->save();
            }
            $stripeIdempotencyKeyObject = new StripeIdempotencyKey();
            $stripeIdempotencyKeyObject->updateIdempotencyKey($cartId, $intent);

            $orderModel = $this->prestashopOrderService->buildOrderModel($psStripePaymentIntent, $intent, $this->context->cart);
            $orderModel = $this->prestashopOrderService->createPsOrder($orderModel);
            StripeProcessLogger::logInfo('Create PrestaShop Order: ' . json_encode($orderModel), 'orderConfirmationReturn', $cartId, $intent->id);

            $this->stripePaymentIntentService->updateStripePaymentIntent($intent->id, ['description' => $orderModel->orderReference]);

            if ((int) Configuration::get(Stripe_official::CAPTURE_WAITING) === (int) $orderModel->status) {
                $stripeCapture = new StripeCapture();
                $stripeCapture->id_payment_intent = $intent->id;
                $stripeCapture->id_order = $orderModel->orderId;
                $stripeCapture->expired = 0;
                $stripeCapture->date_catch = date('Y-m-d H:i:s');
                $stripeCapture->save();
            }

            $this->prestashopOrderService->createPsStripePayment($intent, $orderModel);

            $redirectUrl = $orderModel->order ? $this->prestashopOrderService->getOrderConfirmationLink($orderModel) : $failUrl;
        } catch (Exception $e) {
            StripeProcessLogger::logError('Order Confirmation Error => ' . $e->getMessage() . ' - ' . $e->getTraceAsString(), 'orderConfirmationReturn', $cartId, $intent->id);
        }

        Tools::redirect($redirectUrl);
    }
}
